/**
 * 星聚合登录 JavaScript SDK
 * 
 * @author 星聚合登录
 * @version 1.0.0
 */

class OAuth {
    /**
     * 构造函数
     * @param {Object} config 配置参数
     * @param {string} config.apiurl API地址
     * @param {string} config.appid 应用ID
     * @param {string} config.appkey 应用密钥
     * @param {string} config.callback 回调地址
     */
    constructor(config) {
        this.apiurl = (config.apiurl || '').replace(/\/$/, '');
        this.appid = config.appid || '';
        this.appkey = config.appkey || '';
        this.callback = config.callback || '';
    }

    /**
     * 发起登录请求
     * @param {string} type 登录类型 (qq/wx/alipay/sina/github等)
     * @param {string} state 自定义参数，回调时原样返回
     * @returns {Promise<Object>}
     */
    async login(type, state = '') {
        const params = {
            act: 'login',
            appid: this.appid,
            appkey: this.appkey,
            type: type,
            redirect_uri: this.callback,
            state: state,
        };
        return this.request(params);
    }

    /**
     * 获取用户信息
     * @param {string} code 授权码
     * @returns {Promise<Object>}
     */
    async callback(code) {
        const params = {
            act: 'callback',
            appid: this.appid,
            appkey: this.appkey,
            code: code,
        };
        return this.request(params);
    }

    /**
     * 查询用户信息
     * @param {string} type 登录类型
     * @param {string} socialUid 用户唯一标识
     * @returns {Promise<Object>}
     */
    async query(type, socialUid) {
        const params = {
            act: 'query',
            appid: this.appid,
            appkey: this.appkey,
            type: type,
            social_uid: socialUid,
        };
        return this.request(params);
    }

    /**
     * 发送请求
     * @param {Object} params 请求参数
     * @returns {Promise<Object>}
     */
    async request(params) {
        const query = new URLSearchParams(params).toString();
        const url = `${this.apiurl}/connect.php?${query}`;

        try {
            const response = await fetch(url);
            const data = await response.json();
            return data;
        } catch (error) {
            return { code: -1, msg: error.message };
        }
    }
}

// Node.js 环境导出
if (typeof module !== 'undefined' && module.exports) {
    module.exports = OAuth;
}
